<?php
/*--------------------------------------------------------------
   GambioOrdersTrend.php 2021-08-16
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2021 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
 -------------------------------------------------------------*/

declare(strict_types=1);

namespace Gambio\Admin\Modules\Statistics\App\Overview\Model\Entities\WidgetDefinition;

use Doctrine\DBAL\Connection;
use Gambio\Admin\Modules\Statistics\App\Data\Factory as DataFactory;
use Gambio\Admin\Modules\Statistics\App\Overview\Factory as OverviewFactory;
use Gambio\Admin\Modules\Statistics\App\Overview\Factory\Option\Predefined\TimespanDropdown;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Collections\Options;
use Gambio\Admin\Modules\Statistics\Model\ValueObjects\Data;
use Gambio\Core\Application\ValueObjects\UserPreferences;

class GambioOrdersTrend extends GambioWidgetDefinition
{
    private const TYPE = 'gambio.orders.trend';
    
    private const VERSION = '0.1.0';
    
    private const ICON = "data:image/png;base64,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";
    
    private const WIDGET_TITLE = [
        self::LANGUAGE_CODE_GERMAN  => 'Bestellungen (Trend)',
        self::LANGUAGE_CODE_ENGLISH => 'Orders (Trend)'
    ];
    
    private const CATEGORY_TITLE = [
        self::LANGUAGE_CODE_GERMAN  => 'Bestellungen',
        self::LANGUAGE_CODE_ENGLISH => 'Orders'
    ];
    
    /**
     * @var DataFactory
     */
    private $dataFactory;
    
    /**
     * @var UserPreferences
     */
    private $userPreferences;
    
    /**
     * @var Connection
     */
    private $connection;
    
    
    /**
     * @inheritDoc
     */
    public function __construct(
        OverviewFactory $overviewFactory,
        DataFactory $dataFactory,
        UserPreferences $userPreferences,
        Connection $connection
    ) {
        $this->dataFactory     = $dataFactory;
        $this->userPreferences = $userPreferences;
        $this->connection      = $connection;
        
        parent::__construct($overviewFactory->createType(self::TYPE),
                            $overviewFactory->createVersion(self::VERSION),
                            $overviewFactory->createIconUsingData(self::ICON),
                            $overviewFactory->useVisualizations()->createAreaChart(),
                            $overviewFactory->useOptions()->createOptions($overviewFactory->useOptions()
                                                                              ->usePredefined()
                                                                              ->createTimespanDropdown($overviewFactory)),
                            $overviewFactory->createTitles($overviewFactory->createTitle($overviewFactory->createLanguageCode(self::LANGUAGE_CODE_GERMAN),
                                                                                         self::WIDGET_TITLE[self::LANGUAGE_CODE_GERMAN]),
                                                           $overviewFactory->createTitle($overviewFactory->createLanguageCode(self::LANGUAGE_CODE_ENGLISH),
                                                                                         self::WIDGET_TITLE[self::LANGUAGE_CODE_ENGLISH])));
    }
    
    
    /**
     * @inheritDoc
     */
    public function data(Options $options): Data
    {
        $categories    = [];
        $values        = [];
        $categoryTitle = self::CATEGORY_TITLE[$this->userPreferences->languageId()
                                              === self::LANGUAGE_ID_GERMAN ? self::LANGUAGE_CODE_GERMAN : self::LANGUAGE_CODE_ENGLISH];
        $timespan      = $this->dataFactory->useTimespan()->createFromTerm($options->getById(TimespanDropdown::ID)
                                                                               ->value());
        $orders        = $this->connection->createQueryBuilder()
            ->select([
                         'COUNT(date_purchased) as amount',
                         'UNIX_TIMESTAMP(DATE(date_purchased)) as date'
                     ])
            ->from('orders')
            ->where('orders_status NOT IN (:excludedOrderIds)')
            ->andWhere('date_purchased BETWEEN :startDate AND :endDate')
            ->groupBy('date')
            ->orderBy('date')
            ->setParameters([
                                ':excludedOrderIds' => implode(',', self::EXCLUDED_ORDER_STATUS_IDS),
                                ':startDate'        => $timespan->startDate()
                                    ->format(self::QUERY_TIMESPAN_FORMAT_DATE_START),
                                ':endDate'          => $timespan->endDate()
                                    ->format(self::QUERY_TIMESPAN_FORMAT_DATE_END)
                            ])
            ->execute()
            ->fetchAll();
        
        foreach ($orders as $order) {
            $categories[] = $this->dataFactory->useSerialData()->createCategory($order['date']);
            $values[]     = $this->dataFactory->useSerialData()->createItemValue((int)$order['amount']);
        }
        
        return $this->dataFactory->useSerialData()->createSerialData($this->dataFactory->useSerialData()
                                                                         ->createCategories(...$categories),
                                                                     $this->dataFactory->useSerialData()
                                                                         ->createItems($this->dataFactory->useSerialData()
                                                                                           ->createItem($this->dataFactory->useSerialData()
                                                                                                            ->createItemTitle($categoryTitle),
                                                                                                        $this->dataFactory->useSerialData()
                                                                                                            ->createItemValues(...
                                                                                                                $values))));
    }
}
